function [thetastore, sumdthetastore, meritstore, stdstore] = localization(params,theta0,allspots)
% This function finds the parameters (x0,y0,sigma,N,bg,...) for
% a single 2D-image assuming Gaussian PSF models. 

% (C) Copyright 2018
% All rights reserved
% Department of Imaging Physics
% Faculty of Applied Sciences
% Delft University of Technology
% Delft, The Netherlands   

% parameter settings
[Nypixels Nxpixels Ncfg] = size(allspots);
Nitermax = params.Nitermax;
varfit = params.varfit;
model = params.model;
pixelsize = params.pixelsize;

roisizex = Nxpixels*pixelsize;
roisizey = Nypixels*pixelsize;

% pre-allocation
thetastore = zeros(size(theta0));
sumdthetastore = zeros(Ncfg,Nitermax);
meritstore = zeros(Ncfg,Nitermax);
stdstore = zeros(size(theta0));

for jcfg = 1:Ncfg

% rename the spot
spotscfg = allspots(:,:,jcfg);

% parameters iteration loop
iiter = 1;
tollim = 1e-4;
monitor = 2*tollim;
merit = 1e10;
alamda = 1.0;
alamdafac = 10;

% initial values and max/min values
switch model
  case 'standard2D' 
    x0 = theta0(jcfg,1);
    y0 = theta0(jcfg,2);
    sigma = theta0(jcfg,3);
    Nph = theta0(jcfg,4);
    bg = theta0(jcfg,5);
    thetainit = [x0,y0,sigma,Nph,bg];
    thetamax = [roisizex/2,roisizey/2,min([2.5*pixelsize, 1.5*thetainit(3)]),2*Nph,Nph/Nxpixels/Nypixels/2];
    thetamin = [-roisizex/2,-roisizey/2,0.5*pixelsize,Nph/10,0];
    convergenceweight = [1/pixelsize,1/pixelsize,1/pixelsize,1/Nph,1/Nph];
  case 'astigmatic3D'
    x0 = theta0(jcfg,1);
    y0 = theta0(jcfg,2);
    sigma = theta0(jcfg,3);
    Nph = theta0(jcfg,4);
    bg = theta0(jcfg,5);
    F = theta0(jcfg,6);
    thetainit = [x0,y0,sigma,Nph,bg,F];
    thetamax = [roisizex/2,roisizey/2,min([2.5*pixelsize, 1.5*thetainit(3)]),2*Nph,Nph/Nxpixels/Nypixels/2,1];
    thetamin = [-roisizex/2,-roisizey/2,0.5*pixelsize,Nph/10,0,-1];
    convergenceweight = [1/pixelsize,1/pixelsize,1/pixelsize,1/Nph,1/Nph,1];
  case 'diffractivecolor2D'
    x0 = theta0(jcfg,1);
    y0 = theta0(jcfg,2);
    sigma = theta0(jcfg,3);
    Nph = theta0(jcfg,4);
    bg = theta0(jcfg,5);
    lambda = theta0(jcfg,6);
    thetainit = [x0,y0,sigma,Nph,bg,lambda];
    thetamax = [roisizex/2,roisizey/2,2.5*pixelsize,2*Nph,Nph/Nxpixels/Nypixels/2,750];
    thetamin = [-roisizex/2,-roisizey/2,0.5*pixelsize,Nph/10,0,350];
    convergenceweight = [1/pixelsize,1/pixelsize,1/pixelsize,1/Nph,1/Nph,1];
    if ((thetainit(6)>thetamax(6))||(thetainit(6)<thetamin(6)))  
      thetainit(6) = (thetamax(6)+thetamin(6))/2;
    end
  case 'diffractivecolor3D'
    Fmax = params.Fmax;  
    x0 = theta0(jcfg,1);
    y0 = theta0(jcfg,2);
    sigma = theta0(jcfg,3);
    Nph = theta0(jcfg,4);
    bg = theta0(jcfg,5);
    lambda = theta0(jcfg,6);
    F = theta0(jcfg,7);
    thetainit = [x0,y0,sigma,Nph,bg,lambda,F];
    thetamax = [roisizex/2,roisizey/2,2.5*pixelsize,2*Nph,Nph/Nxpixels/Nypixels/2,750,Fmax];
    thetamin = [-roisizex/2,-roisizey/2,0.5*pixelsize,Nph/10,0,350,-Fmax];
    convergenceweight = [1/pixelsize,1/pixelsize,1/pixelsize,1/Nph,1/Nph,1,1];
    for jk = 6:7
      if ((thetainit(jk)>thetamax(jk))||(thetainit(jk)<thetamin(jk)))  
        thetainit(jk) = (thetamax(jk)+thetamin(jk))/2;
      end
    end
end

% start iteration loop 
theta = thetainit;
thetaretry = thetainit;

while ((iiter<=Nitermax) && (monitor>tollim))
  
  thetaprev = theta;
  meritprev = merit;
  
  % calculate Poisson-rates and derivatives, and subsequently the log-likelihood and derivatives.
  [mu,dmudtheta] = poissonrate(params,theta);
  [merit,grad,Hessian] = likelihood(spotscfg,mu,dmudtheta,varfit);
  
  % Levenberg-Marquardt algorithm
  dmerit = -1.0;
  trustitermax = 10;
  trustiter = 0;
  while ((dmerit<0) && (trustiter<=trustitermax))
  % check for det(H)=0 in order to avoid inversion of H in limit alamda to zero
    if (abs(det(Hessian))<10*eps)
      dmerit = -1.0;
      thetatry = theta;
      merittry = merit;
    else  
  % update parameters 
      thetatry = thetaupdate(theta,thetamax,thetamin,thetaretry,grad,Hessian,alamda);
  % calculate update merit function
      [mu,dmudtheta] = poissonrate(params,theta);
      [merittry,grad,Hessian] = likelihood(spotscfg,mu,dmudtheta,varfit);
      dmerit = merittry-merit;
    end  
   % modify Levenberg-Marquardt parameter
    if (dmerit<0)
      alamda = alamdafac*alamda;
    else
      alamda = alamda/alamdafac;
    end
    trustiter = trustiter+1;
  end
  theta = thetatry;
  merit = merittry;
   
  % monitor convergence of iteration
  dtheta = theta-thetaprev;
  sumdtheta = sqrt(sum((convergenceweight.*dtheta).^2));
  dmerit = merit-meritprev;
  monitor = abs(dmerit/merit);
  
  sumdthetastore(jcfg,iiter) = sumdtheta;
  meritstore(jcfg,iiter) = merit;
  
  % update counter  
  iiter = iiter+1;
  
end
 
% calculation of covariance matrix and estimated accuracy at found parameter-values
  [mu,dmudtheta] = poissonrate(params,theta);
  [merit,grad,Hessian] = likelihood(spotscfg,mu,dmudtheta,varfit);
  for jiter = iiter:Nitermax
    meritstore(jcfg,jiter) = merit;
  end
  stdstore(jcfg,:) = diag(sqrt(-inv(Hessian)));
  
% store found fitted parameters.
  thetastore(jcfg,:) = theta;
end


